import os
import tkinter as tk
from tkinter import messagebox
import fitz  # PyMuPDF

# FOLDER PATHS
PROJECT_FOLDER = r"C:\Users\conta\Downloads\AI ICAI\AI LEVEL 2\DAY 3\Venkatredactpdf"
INPUT_FOLDER = os.path.join(PROJECT_FOLDER, "inputfile")
OUTPUT_FOLDER = os.path.join(PROJECT_FOLDER, "outputfile")


class PDFRedactorApp:
    def __init__(self, root):
        self.root = root
        self.root.title("PDF Redaction App")
        self.root.geometry("1000x700")

        # --- Toolbar with Navigation, Zoom, Clear, Save ---
        toolbar = tk.Frame(root)
        toolbar.pack(fill=tk.X, side=tk.TOP)

        tk.Button(toolbar, text="Previous Page", command=self.prev_page).pack(side=tk.LEFT, padx=5, pady=5)
        tk.Button(toolbar, text="Next Page", command=self.next_page).pack(side=tk.LEFT, padx=5)
        tk.Button(toolbar, text="Zoom In", command=lambda: self.zoom_page(1.25)).pack(side=tk.LEFT, padx=5)
        tk.Button(toolbar, text="Zoom Out", command=lambda: self.zoom_page(0.8)).pack(side=tk.LEFT, padx=5)
        tk.Button(toolbar, text="Clear Marks", command=self.clear_marks).pack(side=tk.LEFT, padx=5)
        tk.Button(toolbar, text="Save Redacted PDF", command=self.save_all_pdfs).pack(side=tk.LEFT, padx=5)

        # --- Canvas with Scrollbars ---
        self.canvas_frame = tk.Frame(root)
        self.canvas_frame.pack(fill=tk.BOTH, expand=True)

        self.canvas = tk.Canvas(self.canvas_frame, bg="white")
        self.canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        self.v_scroll = tk.Scrollbar(self.canvas_frame, orient=tk.VERTICAL, command=self.canvas.yview)
        self.v_scroll.pack(side=tk.RIGHT, fill=tk.Y)
        self.h_scroll = tk.Scrollbar(root, orient=tk.HORIZONTAL, command=self.canvas.xview)
        self.h_scroll.pack(side=tk.BOTTOM, fill=tk.X)

        self.canvas.configure(yscrollcommand=self.v_scroll.set, xscrollcommand=self.h_scroll.set)

        # Canvas bindings
        self.canvas.bind("<ButtonPress-1>", self.on_draw_start)
        self.canvas.bind("<B1-Motion>", self.on_draw_drag)
        self.canvas.bind("<ButtonRelease-1>", self.on_draw_end)

        # --- Variables ---
        self.pdf_doc = fitz.open()  # combined PDF
        self.original_files = []  # list of original files for saving separately
        self.page_index = 0
        self.start_x = self.start_y = None
        self.rect = None
        self.redactions_per_page = {}  # store rects per page
        self.img = None
        self.zoom_factor = 1.0

        # --- Load PDFs automatically from input folder ---
        self.load_input_pdfs()

    # --- Load all PDFs from input folder ---
    def load_input_pdfs(self):
        pdf_files = [f for f in os.listdir(INPUT_FOLDER) if f.lower().endswith(".pdf")]
        if not pdf_files:
            messagebox.showwarning("No PDFs", "No PDF files found in input folder!")
            return

        self.pdf_doc = fitz.open()  # new empty PDF
        self.original_files = []

        for pdf_file in pdf_files:
            path = os.path.join(INPUT_FOLDER, pdf_file)
            doc = fitz.open(path)
            self.pdf_doc.insert_pdf(doc)
            self.original_files.append((pdf_file, doc))  # keep original doc for saving separately

        self.page_index = 0
        self.redactions_per_page = {}
        messagebox.showinfo("PDFs Loaded", f"Loaded {len(pdf_files)} PDFs as one document.")
        self.display_page()

    # --- Display current page ---
    def display_page(self):
        if self.pdf_doc:
            self.canvas.delete("all")
            page = self.pdf_doc[self.page_index]
            mat = fitz.Matrix(self.zoom_factor, self.zoom_factor)
            pix = page.get_pixmap(matrix=mat)
            self.img = tk.PhotoImage(data=pix.tobytes("ppm"))
            self.canvas.create_image(0, 0, anchor="nw", image=self.img)
            self.canvas.config(scrollregion=self.canvas.bbox(tk.ALL))

            # Draw existing redactions
            for rect_coords in self.redactions_per_page.get(self.page_index, []):
                self.canvas.create_rectangle(*rect_coords, fill="black")

    # --- Page navigation ---
    def prev_page(self):
        if self.pdf_doc and self.page_index > 0:
            self.page_index -= 1
            self.display_page()

    def next_page(self):
        if self.pdf_doc and self.page_index < len(self.pdf_doc) - 1:
            self.page_index += 1
            self.display_page()

    # --- Zoom ---
    def zoom_page(self, factor):
        self.zoom_factor *= factor
        self.display_page()

    # --- Drawing ---
    def on_draw_start(self, event):
        self.start_x = self.canvas.canvasx(event.x)
        self.start_y = self.canvas.canvasy(event.y)
        self.rect = self.canvas.create_rectangle(self.start_x, self.start_y,
                                                 self.start_x, self.start_y, outline="red")

    def on_draw_drag(self, event):
        if self.rect:
            x = self.canvas.canvasx(event.x)
            y = self.canvas.canvasy(event.y)
            self.canvas.coords(self.rect, self.start_x, self.start_y, x, y)

    def on_draw_end(self, event):
        if self.rect:
            x = self.canvas.canvasx(event.x)
            y = self.canvas.canvasy(event.y)
            coords = (self.start_x, self.start_y, x, y)
            self.redactions_per_page.setdefault(self.page_index, []).append(coords)
            self.canvas.delete(self.rect)
            self.canvas.create_rectangle(*coords, fill="black")
            self.rect = None

    # --- Clear marks ---
    def clear_marks(self):
        if self.page_index in self.redactions_per_page:
            del self.redactions_per_page[self.page_index]
        self.display_page()

    # --- Save redacted PDFs separately ---
    def save_all_pdfs(self):
        if not self.original_files:
            messagebox.showwarning("No PDFs", "No PDFs loaded to save!")
            return

        for idx, (file_name, original_doc) in enumerate(self.original_files):
            for page_num in range(len(original_doc)):
                global_page_num = sum(len(doc) for _, doc in self.original_files[:idx]) + page_num
                page = original_doc[page_num]

                if global_page_num in self.redactions_per_page:
                    rects = self.redactions_per_page[global_page_num]
                    for x1, y1, x2, y2 in rects:
                        # Convert canvas coordinates to PDF coordinates properly using zoom_factor
                        pdf_x1 = x1 / self.zoom_factor
                        pdf_y1 = y1 / self.zoom_factor
                        pdf_x2 = x2 / self.zoom_factor
                        pdf_y2 = y2 / self.zoom_factor
                        rect = fitz.Rect(pdf_x1, pdf_y1, pdf_x2, pdf_y2)
                        page.add_redact_annot(rect, fill=(0, 0, 0))
                    # Apply redactions to make black boxes permanent
                    page.apply_redactions()

            # Save each file separately
            save_path = os.path.join(OUTPUT_FOLDER, file_name)
            original_doc.save(save_path, garbage=4, deflate=True)

        messagebox.showinfo("Saved", f"Redacted PDFs saved in {OUTPUT_FOLDER}")


if __name__ == "__main__":
    root = tk.Tk()
    app = PDFRedactorApp(root)
    root.mainloop()
